# WizStudio Status - Windows Server Monitoring and API Service Installation Script
# Name: status_vps_windows.ps1
# Version: 1.0.0 (2025-01-03)
# Features:
# 1) Interactive menu: Install(default)/Update/View UUID/Change Token/Diagnostic
# 2) Generate UUID, configure Token, select network adapter and port(default 8008), deploy Windows service
# 3) API access via http://IP:PORT/path?uuid=&token=, returns JSON string
# 4) Echo service management commands after installation
# 5) Support Windows system monitoring (CPU/Memory/Disk/Network)
# 6) Diagnostic mode for troubleshooting
#
# Config file: wizstudio_status.conf
# Install directory: current directory
# Service program: server.py

param(
    [string]$Action = "menu"
)

# Global variables
$INSTALL_DIR = $PSScriptRoot
$CONFIG_FILE = Join-Path $INSTALL_DIR "wizstudio_status.conf"
$SERVER_FILE = Join-Path $INSTALL_DIR "server.py"
$DEFAULT_PORT = "8008"

# Print title
function Show-Title {
    Write-Host "=============================== " -ForegroundColor Cyan
    Write-Host " WizStudio Status Windows Edition " -ForegroundColor Cyan
    Write-Host "=============================== " -ForegroundColor Cyan
}

# Read input with default value
function Read-InputWithDefault {
    param(
        [string]$Prompt,
        [string]$Default
    )
    
    $input = Read-Host "$Prompt [$Default]"
    if ([string]::IsNullOrEmpty($input)) {
        return $Default
    }
    return $input
}

# Generate UUID
function New-UUID {
    return [System.Guid]::NewGuid().ToString()
}

# Get network adapters
function Get-NetworkAdapters {
    $adapters = @()
    
    try {
        # Method 1: Try Get-NetAdapter (Windows 8/2012+)
        $netAdapters = Get-NetAdapter -ErrorAction SilentlyContinue | Where-Object { 
            $_.Status -eq "Up" -and 
            $_.Name -notlike "*Loopback*" -and 
            $_.Name -notlike "*Teredo*" -and
            $_.Name -notlike "*isatap*" -and
            $_.InterfaceType -ne 24  # Exclude loopback
        }
        
        if ($netAdapters) {
            $adapters = $netAdapters
        }
    }
    catch {
        Write-Host "Get-NetAdapter not available, trying WMI method..." -ForegroundColor Yellow
    }
    
    # Method 2: Fallback to WMI (Windows 7/2008 R2 and older)
    if ($adapters.Count -eq 0) {
        try {
            $wmiAdapters = Get-WmiObject -Class Win32_NetworkAdapter -ErrorAction SilentlyContinue | Where-Object {
                $_.NetConnectionStatus -eq 2 -and  # Connected
                $_.Name -notlike "*Loopback*" -and
                $_.Name -notlike "*Teredo*" -and
                $_.Name -notlike "*isatap*" -and
                $_.AdapterType -notlike "*Loopback*"
            }
            
            if ($wmiAdapters) {
                $adapters = $wmiAdapters | ForEach-Object {
                    [PSCustomObject]@{
                        Name = $_.Name
                        InterfaceIndex = $_.InterfaceIndex
                        Status = "Up"
                        MacAddress = $_.MACAddress
                    }
                }
            }
        }
        catch {
            Write-Host "WMI method also failed, trying ipconfig..." -ForegroundColor Yellow
        }
    }
    
    # Method 3: Parse ipconfig output as last resort
    if ($adapters.Count -eq 0) {
        try {
            $ipconfig = ipconfig /all
            $currentAdapter = $null
            $adapterList = @()
            
            foreach ($line in $ipconfig) {
                if ($line -match "^([^:]+adapter[^:]*):$") {
                    if ($currentAdapter -and $currentAdapter.HasIP) {
                        $adapterList += $currentAdapter
                    }
                    $currentAdapter = [PSCustomObject]@{
                        Name = $matches[1].Trim()
                        HasIP = $false
                        InterfaceIndex = $adapterList.Count + 1
                        Status = "Up"
                    }
                }
                elseif ($line -match "IPv4.*: (.+)" -and $currentAdapter) {
                    $ip = $matches[1].Trim()
                    if ($ip -notmatch "^169\.254\." -and $ip -ne "127.0.0.1") {
                        $currentAdapter.HasIP = $true
                    }
                }
            }
            
            if ($currentAdapter -and $currentAdapter.HasIP) {
                $adapterList += $currentAdapter
            }
            
            $adapters = $adapterList | Where-Object { 
                $_.Name -notlike "*Loopback*" -and 
                $_.Name -notlike "*Teredo*" -and
                $_.Name -notlike "*isatap*"
            }
        }
        catch {
            Write-Host "All network adapter detection methods failed." -ForegroundColor Red
        }
    }
    
    return $adapters
}

# Select network adapter
function Select-NetworkAdapter {
    Write-Host "Detecting network adapters..." -ForegroundColor Yellow
    $adapters = Get-NetworkAdapters
    
    # Debug information
    Write-Host "Debug: Found $($adapters.Count) adapters" -ForegroundColor Magenta
    if ($adapters.Count -gt 0) {
        Write-Host "Debug: Adapter types: $($adapters | ForEach-Object { $_.GetType().Name } | Select-Object -Unique)" -ForegroundColor Magenta
    }
    
    # Handle null or empty adapters
    if (-not $adapters -or $adapters.Count -eq 0) {
        Write-Host "No available network adapters detected." -ForegroundColor Red
        Write-Host "This might be due to:" -ForegroundColor Yellow
        Write-Host "1. No active network connections" -ForegroundColor White
        Write-Host "2. PowerShell version compatibility issues" -ForegroundColor White
        Write-Host "3. Insufficient permissions" -ForegroundColor White
        Write-Host ""
        Write-Host "Trying alternative detection method..." -ForegroundColor Cyan
        
        # Try simple ipconfig method
        try {
            $ipconfig = ipconfig
            Write-Host "Available connections from ipconfig:" -ForegroundColor Yellow
            $connectionFound = $false
            foreach ($line in $ipconfig) {
                if ($line -match "adapter (.+):$") {
                    $adapterName = $matches[1]
                    if ($adapterName -notlike "*Loopback*") {
                        Write-Host "  - $adapterName" -ForegroundColor Green
                        $connectionFound = $true
                    }
                }
            }
            
            if ($connectionFound) {
                Write-Host ""
                $manualName = Read-Host "Enter the exact adapter name from above (or 'external' for KVM)"
                if ([string]::IsNullOrEmpty($manualName)) {
                    return "external"
                }
                return $manualName
            }
        }
        catch {
            Write-Host "ipconfig method also failed." -ForegroundColor Red
        }
        
        Write-Host "You can manually enter the adapter name:" -ForegroundColor Cyan
        Write-Host "For KVM virtual machines, try: external" -ForegroundColor Yellow
        $manualName = Read-Host "Enter adapter name (or press Enter for 'external')"
        if ([string]::IsNullOrEmpty($manualName)) {
            return "external"
        }
        return $manualName
    }
    
    Write-Host "Available network adapters:" -ForegroundColor Yellow
    
    # Ensure adapters is an array
    if ($adapters -isnot [array]) {
        $adapters = @($adapters)
    }
    
    for ($i = 0; $i -lt $adapters.Count; $i++) {
        $adapter = $adapters[$i]
        $ip = "Unknown"
        
        Write-Host "Debug: Processing adapter $i : $($adapter.Name)" -ForegroundColor Magenta
        
        # Try to get IP address
        try {
            if ($adapter.InterfaceIndex) {
                $ipAddr = Get-NetIPAddress -InterfaceIndex $adapter.InterfaceIndex -AddressFamily IPv4 -ErrorAction SilentlyContinue
                if ($ipAddr) {
                    $ip = $ipAddr.IPAddress
                }
            }
        }
        catch {
            # Try WMI method for IP
            try {
                $wmiConfig = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -ErrorAction SilentlyContinue | 
                    Where-Object { $_.Description -eq $adapter.Name -and $_.IPAddress }
                if ($wmiConfig -and $wmiConfig.IPAddress) {
                    $ip = $wmiConfig.IPAddress[0]
                }
            }
            catch {
                $ip = "No IPv4"
            }
        }
        
        Write-Host "  $($i + 1)) $($adapter.Name) [$ip]" -ForegroundColor Green
    }
    
    Write-Host ""
    Write-Host "Total adapters found: $($adapters.Count)" -ForegroundColor Cyan
    
    do {
        $choice = Read-Host "Please select network adapter (enter number 1-$($adapters.Count))"
        
        if ([string]::IsNullOrEmpty($choice)) {
            Write-Host "Please enter a number between 1 and $($adapters.Count)" -ForegroundColor Red
            continue
        }
        
        if ($choice -match '^\d+$') {
            $choiceNum = [int]$choice
            if ($choiceNum -ge 1 -and $choiceNum -le $adapters.Count) {
                $selectedAdapter = $adapters[$choiceNum - 1]
                Write-Host "Selected: $($selectedAdapter.Name)" -ForegroundColor Green
                return $selectedAdapter.Name
            }
        }
        
        Write-Host "Invalid selection. Please enter a number between 1 and $($adapters.Count)" -ForegroundColor Red
    } while ($true)
}

# Write configuration file
function Write-Config {
    param(
        [string]$Port,
        [string]$UUID,
        [string]$Token,
        [string]$Interface,
        [string]$Comment
    )
    
    $configContent = @"
PORT=$Port
UUID=$UUID
TOKEN=$Token
IFACE=$Interface
COMMENT=$Comment
"@
    
    $configContent | Out-File -FilePath $CONFIG_FILE -Encoding UTF8
}

# Check if Python is installed
function Test-PythonInstalled {
    try {
        $pythonVersion = python --version 2>&1
        if ($pythonVersion -match "Python") {
            return $true
        }
    }
    catch {
        return $false
    }
    return $false
}

# Install dependencies
function Install-Dependencies {
    if (-not (Test-PythonInstalled)) {
        Write-Host "Python not detected. Please install Python 3.7 or higher first." -ForegroundColor Red
        Write-Host "Download URL: https://www.python.org/downloads/" -ForegroundColor Yellow
        return $false
    }
    
    Write-Host "Python is installed, checking dependencies..." -ForegroundColor Green
    return $true
}

# Write Python server program
function Write-ServerPy {
    $serverContent = @'
#!/usr/bin/env python3
# -*- coding: utf-8 -*-
import json
import os
import platform
import subprocess
import time
import psutil
from http.server import BaseHTTPRequestHandler, HTTPServer
from urllib.parse import urlparse, parse_qs

CONFIG_FILE = "wizstudio_status.conf"

def read_config():
    cfg = {}
    try:
        if os.path.exists(CONFIG_FILE):
            with open(CONFIG_FILE, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    if "=" in line:
                        k, v = line.split("=", 1)
                        cfg[k.strip()] = v.strip()
    except Exception as e:
        print(f"Error reading config: {e}")
    return cfg

def human_gb(b, digits=2):
    try:
        return f"{round(b / (1024**3), digits)}GB"
    except Exception:
        return "0GB"

def human_percent(v, digits=1):
    try:
        return f"{round(v, digits)}%"
    except Exception:
        return "0%"

def human_bitrate(bps):
    try:
        if bps >= 1_000_000:
            return f"{round(bps/1_000_000.0, 2)}Mbps"
        else:
            return f"{round(bps/1000.0, 2)}Kbps"
    except Exception:
        return "0.00Kbps"

def cpu_info():
    try:
        # Get CPU information
        cpu_name = platform.processor()
        if not cpu_name or cpu_name.strip() == "":
            # Try to get from registry
            try:
                import winreg
                key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, 
                    r"HARDWARE\DESCRIPTION\System\CentralProcessor\0")
                cpu_name = winreg.QueryValueEx(key, "ProcessorNameString")[0]
                winreg.CloseKey(key)
            except:
                cpu_name = "Unknown CPU"
        
        # Get core count
        cores = psutil.cpu_count(logical=False)
        threads = psutil.cpu_count(logical=True)
        
        # Get frequency
        try:
            freq = psutil.cpu_freq()
            if freq and freq.max:
                freq_str = f"@{round(freq.max / 1000.0, 2)}GHz"
            else:
                freq_str = ""
        except:
            freq_str = ""
        
        core_part = f" {cores}Cores" if cores else ""
        if threads and threads != cores:
            core_part += f"/{threads}Threads"
        
        return f"{cpu_name.strip()}{core_part}{freq_str}"
    except Exception as e:
        return f"Unknown CPU (Error: {e})"

def cpu_usage_percent():
    try:
        return psutil.cpu_percent(interval=1.0)
    except Exception:
        return 0.0

def mem_usage():
    try:
        mem = psutil.virtual_memory()
        return (mem.used, mem.total)
    except Exception:
        return (0, 0)

def disk_usage():
    try:
        total_used = 0
        total_size = 0
        
        # Get all disk partitions
        partitions = psutil.disk_partitions()
        for partition in partitions:
            try:
                if partition.fstype:  # Only count partitions with filesystem
                    usage = psutil.disk_usage(partition.mountpoint)
                    total_used += usage.used
                    total_size += usage.total
            except (PermissionError, FileNotFoundError):
                continue
        
        return (total_used, total_size)
    except Exception:
        return (0, 0)

def disk_io():
    try:
        # First sample
        io1 = psutil.disk_io_counters()
        if not io1:
            return (0.0, 0.0, 0, 0)
        
        time.sleep(1.0)
        
        # Second sample
        io2 = psutil.disk_io_counters()
        if not io2:
            return (0.0, 0.0, 0, 0)
        
        # Calculate differences
        read_bytes = max(0, io2.read_bytes - io1.read_bytes)
        write_bytes = max(0, io2.write_bytes - io1.write_bytes)
        read_count = max(0, io2.read_count - io1.read_count)
        write_count = max(0, io2.write_count - io1.write_count)
        
        # Convert to MB/s
        read_mbps = read_bytes / (1024 * 1024)
        write_mbps = write_bytes / (1024 * 1024)
        
        return (read_mbps, write_mbps, read_count, write_count)
    except Exception:
        return (0.0, 0.0, 0, 0)

def network_io(interface_name):
    try:
        # Get network interface statistics
        net_io = psutil.net_io_counters(pernic=True)
        
        # Find matching interface
        target_interface = None
        for iface_name, stats in net_io.items():
            if interface_name.lower() in iface_name.lower():
                target_interface = stats
                break
        
        if not target_interface:
            # If specified interface not found, use total
            target_interface = psutil.net_io_counters()
        
        if not target_interface:
            return (0, 0)
        
        # First sample
        bytes_sent1 = target_interface.bytes_sent
        bytes_recv1 = target_interface.bytes_recv
        
        time.sleep(2.0)
        
        # Second sample
        if interface_name:
            net_io2 = psutil.net_io_counters(pernic=True)
            target_interface2 = None
            for iface_name, stats in net_io2.items():
                if interface_name.lower() in iface_name.lower():
                    target_interface2 = stats
                    break
            if not target_interface2:
                target_interface2 = psutil.net_io_counters()
        else:
            target_interface2 = psutil.net_io_counters()
        
        if not target_interface2:
            return (0, 0)
        
        bytes_sent2 = target_interface2.bytes_sent
        bytes_recv2 = target_interface2.bytes_recv
        
        # Calculate rate (bps)
        upload_bps = max(0, (bytes_sent2 - bytes_sent1) * 8.0 / 2.0)
        download_bps = max(0, (bytes_recv2 - bytes_recv1) * 8.0 / 2.0)
        
        return (upload_bps, download_bps)
    except Exception as e:
        print(f"Network IO error: {e}")
        return (0, 0)

def get_uptime_formatted():
    try:
        # Windows system boot time
        boot_time = psutil.boot_time()
        uptime_seconds = time.time() - boot_time
        
        # Convert to time units
        total_minutes = int(uptime_seconds // 60)
        total_hours = total_minutes // 60
        total_days = total_hours // 24
        years = total_days // 365
        months = (total_days % 365) // 30
        
        minutes = total_minutes % 60
        hours = total_hours % 24
        days = total_days % 30
        
        # Build formatted string
        parts = []
        if years > 0:
            parts.append(f"{years}y")
        if months > 0:
            parts.append(f"{months}m")
        if days > 0:
            parts.append(f"{days}d")
        if hours > 0:
            parts.append(f"{hours}h")
        if minutes > 0:
            parts.append(f"{minutes}min")
        
        if parts:
            return "".join(parts)
        else:
            return "less than 1min"
    except Exception as e:
        return f"N/A (Error: {e})"

class Handler(BaseHTTPRequestHandler):
    server_version = "WizStudioStatus/1.0"

    def _send(self, code, payload_str):
        if isinstance(payload_str, dict):
            body = json.dumps(payload_str, ensure_ascii=False).encode("utf-8")
        else:
            body = json.dumps(payload_str, ensure_ascii=False).encode("utf-8")
        
        self.send_response(code)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(body)))
        self.end_headers()
        self.wfile.write(body)

    def do_GET(self):
        cfg = read_config()
        req = urlparse(self.path)
        q = parse_qs(req.query)
        uuid = q.get("uuid", [""])[0]
        token = q.get("token", [""])[0]

        want_uuid = cfg.get("UUID", "")
        want_token = cfg.get("TOKEN", "")
        iface = cfg.get("IFACE", "")

        # Check diagnostic mode
        is_diagnostic = "diagnostic" in q and q["diagnostic"][0].lower() in ("1", "true", "yes")
        
        # Authentication
        if not uuid or not token:
            self._send(400, "Missing uuid or token")
            return
        if uuid != want_uuid or token != want_token:
            self._send(401, "Unauthorized")
            return

        path = req.path.rstrip("/")
        try:
            # Diagnostic endpoint
            if path == "/diagnostic" and is_diagnostic:
                diagnostic_data = self._collect_diagnostic_data()
                self._send(200, diagnostic_data)
                return
                
            # Standard API endpoints
            if path == "/cpu/info":
                val = cpu_info()
                self._send(200, val)
            elif path == "/cpu/usage":
                val = f"{round(cpu_usage_percent(), 1)}%"
                self._send(200, val)
            elif path == "/ram/usage0":
                used, total = mem_usage()
                val = f"{human_gb(used, 2)}/{human_gb(total, 2)}"
                self._send(200, val)
            elif path == "/ram/usage1":
                used, total = mem_usage()
                pct = (used/total*100.0) if total > 0 else 0.0
                val = human_percent(pct, 1)
                self._send(200, val)
            elif path == "/disk/usage0":
                used, total = disk_usage()
                val = f"{human_gb(used, 1)}/{human_gb(total, 1)}"
                self._send(200, val)
            elif path == "/disk/usage1":
                used, total = disk_usage()
                pct = (used/total*100.0) if total > 0 else 0.0
                val = f"{round(pct, 1)}%"
                self._send(200, val)
            elif path == "/disk/io":
                read_mbps, write_mbps, read_iops, write_iops = disk_io()
                val = {
                    "read": f"{round(read_mbps, 2)}MB/s",
                    "write": f"{round(write_mbps, 2)}MB/s",
                    "read_iops": read_iops,
                    "write_iops": write_iops
                }
                self._send(200, val)
            elif path == "/network/status":
                if not iface:
                    self._send(500, "No IFACE configured")
                    return
                
                upload_bps, download_bps = network_io(iface)
                val = f"{human_bitrate(upload_bps)}/{human_bitrate(download_bps)}"
                self._send(200, val)
            elif path == "/time":
                val = get_uptime_formatted()
                self._send(200, val)
            else:
                self._send(404, "Not Found")
        except Exception as e:
            self._send(500, f"Error: {e}")

    def _collect_diagnostic_data(self):
        """Collect diagnostic data for troubleshooting"""
        data = {
            "version": "1.0",
            "timestamp": time.time(),
            "system_info": {},
            "cpu": {},
            "memory": {},
            "disk": {},
            "network": {},
            "available_tools": {},
            "debug_info": {}
        }
        
        # System information
        try:
            data["system_info"]["hostname"] = platform.node()
            data["system_info"]["platform"] = platform.platform()
            data["system_info"]["system"] = platform.system()
            data["system_info"]["release"] = platform.release()
            data["system_info"]["version"] = platform.version()
            data["system_info"]["machine"] = platform.machine()
            data["system_info"]["processor"] = platform.processor()
        except Exception as e:
            data["system_info"]["error"] = str(e)
            
        # CPU information
        try:
            data["cpu"]["info"] = cpu_info()
            data["cpu"]["usage"] = f"{round(cpu_usage_percent(), 1)}%"
            data["cpu"]["count_logical"] = psutil.cpu_count(logical=True)
            data["cpu"]["count_physical"] = psutil.cpu_count(logical=False)
        except Exception as e:
            data["cpu"]["error"] = str(e)
            
        # Memory information
        try:
            used, total = mem_usage()
            data["memory"]["used"] = human_gb(used, 2)
            data["memory"]["total"] = human_gb(total, 2)
            data["memory"]["percent"] = human_percent((used/total*100.0) if total > 0 else 0.0, 1)
        except Exception as e:
            data["memory"]["error"] = str(e)
            
        # Disk information
        try:
            used, total = disk_usage()
            data["disk"]["used"] = human_gb(used, 1)
            data["disk"]["total"] = human_gb(total, 1)
            data["disk"]["percent"] = f"{round((used/total*100.0) if total > 0 else 0.0, 1)}%"
            
            # Disk partition information
            partitions = []
            for partition in psutil.disk_partitions():
                try:
                    usage = psutil.disk_usage(partition.mountpoint)
                    partitions.append({
                        "device": partition.device,
                        "mountpoint": partition.mountpoint,
                        "fstype": partition.fstype,
                        "total": human_gb(usage.total, 1),
                        "used": human_gb(usage.used, 1),
                        "free": human_gb(usage.free, 1)
                    })
                except (PermissionError, FileNotFoundError):
                    continue
            data["disk"]["partitions"] = partitions
        except Exception as e:
            data["disk"]["error"] = str(e)
            
        # Network information
        try:
            cfg = read_config()
            iface = cfg.get("IFACE", "")
            data["network"]["configured_iface"] = iface
            
            # Network interface information
            interfaces = []
            net_io = psutil.net_io_counters(pernic=True)
            for name, stats in net_io.items():
                interfaces.append({
                    "name": name,
                    "bytes_sent": stats.bytes_sent,
                    "bytes_recv": stats.bytes_recv,
                    "packets_sent": stats.packets_sent,
                    "packets_recv": stats.packets_recv
                })
            data["network"]["interfaces"] = interfaces
        except Exception as e:
            data["network"]["error"] = str(e)
            
        # Available tools check
        data["available_tools"]["psutil"] = True
        data["available_tools"]["platform"] = True
        
        # Debug information
        try:
            data["debug_info"]["python_version"] = platform.python_version()
            data["debug_info"]["psutil_version"] = psutil.__version__
        except Exception:
            pass
            
        return data

    def log_message(self, fmt, *args):
        return

def run():
    cfg = read_config()
    port = int(cfg.get("PORT", "8008") or "8008")
    addr = ("0.0.0.0", port)
    httpd = HTTPServer(addr, Handler)
    print(f"WizStudio Status Server started on port {port}")
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        print("\nServer stopped")
    finally:
        httpd.server_close()

if __name__ == "__main__":
    run()
'@
    
    $serverContent | Out-File -FilePath $SERVER_FILE -Encoding UTF8
}

# Install service
function Install-Service {
    Write-Host ""
    $confirm = Read-Host "Do you want to start installation? (Y/N) [Y]"
    if ($confirm -and $confirm.ToUpper() -ne "Y") {
        Write-Host "Installation cancelled." -ForegroundColor Yellow
        return
    }
    
    # Port
    $port = Read-InputWithDefault "Please enter service port" $DEFAULT_PORT
    
    # UUID
    $uuid = ""
    if (Test-Path $CONFIG_FILE) {
        $config = Get-Content $CONFIG_FILE -ErrorAction SilentlyContinue
        foreach ($line in $config) {
            if ($line -match "^UUID=(.+)$") {
                $uuid = $matches[1]
                break
            }
        }
    }
    
    if (-not $uuid) {
        $uuid = New-UUID
    }
    Write-Host "Generated/Reused UUID: $uuid" -ForegroundColor Green
    
    # Token
    $token = Read-Host "Please enter Token"
    if ([string]::IsNullOrEmpty($token)) {
        Write-Host "Token cannot be empty." -ForegroundColor Red
        return
    }
    
    # Comment
    $comment = Read-InputWithDefault "Please enter comment" "Default"
    
    # Select network adapter
    Write-Host ""
    $interface = Select-NetworkAdapter
    if (-not $interface) {
        Write-Host "Network adapter selection failed." -ForegroundColor Red
        return
    }
    Write-Host "Selected network adapter: $interface" -ForegroundColor Green
    
    # Check dependencies
    if (-not (Install-Dependencies)) {
        return
    }
    
    # Install psutil
    Write-Host "Installing Python dependencies..." -ForegroundColor Yellow
    try {
        & python -m pip install psutil --quiet
        Write-Host "Dependencies installed successfully." -ForegroundColor Green
    }
    catch {
        Write-Host "Warning: Unable to install psutil, please install manually: pip install psutil" -ForegroundColor Yellow
    }
    
    # Write server program
    Write-ServerPy
    Write-Host "Server program created." -ForegroundColor Green
    
    # Write configuration
    Write-Config $port $uuid $token $interface $comment
    Write-Host "Configuration file created." -ForegroundColor Green
    
    Write-Host ""
    Write-Host "Installation completed!" -ForegroundColor Green
    Write-Host "Start service command: python server.py" -ForegroundColor Cyan
    Write-Host "API example: http://localhost:$port/cpu/info?uuid=$uuid`&token=<your_token>" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "To run service in background, use one of the following methods:" -ForegroundColor Yellow
    Write-Host "1. Use nssm to create Windows service" -ForegroundColor White
    Write-Host "2. Use Task Scheduler for startup" -ForegroundColor White
    Write-Host "3. Run manually: python server.py" -ForegroundColor White
}

# Show UUID
function Show-UUID {
    if (-not (Test-Path $CONFIG_FILE)) {
        Write-Host "Not installed, configuration file not found: $CONFIG_FILE" -ForegroundColor Red
        return
    }
    
    $config = Get-Content $CONFIG_FILE -ErrorAction SilentlyContinue
    $uuid = ""
    foreach ($line in $config) {
        if ($line -match "^UUID=(.+)$") {
            $uuid = $matches[1]
            break
        }
    }
    
    if ($uuid) {
        Write-Host "Current UUID: $uuid" -ForegroundColor Green
    } else {
        Write-Host "UUID not found in configuration." -ForegroundColor Red
    }
}

# Change Token
function Change-Token {
    if (-not (Test-Path $CONFIG_FILE)) {
        Write-Host "Not installed, configuration file not found: $CONFIG_FILE" -ForegroundColor Red
        return
    }
    
    $newToken = Read-Host "Please enter new Token"
    if ([string]::IsNullOrEmpty($newToken)) {
        Write-Host "Token cannot be empty." -ForegroundColor Red
        return
    }
    
    # Read existing configuration
    $config = Get-Content $CONFIG_FILE -ErrorAction SilentlyContinue
    $newConfig = @()
    $tokenUpdated = $false
    
    foreach ($line in $config) {
        if ($line -match "^TOKEN=") {
            $newConfig += "TOKEN=$newToken"
            $tokenUpdated = $true
        } else {
            $newConfig += $line
        }
    }
    
    if (-not $tokenUpdated) {
        $newConfig += "TOKEN=$newToken"
    }
    
    $newConfig | Out-File -FilePath $CONFIG_FILE -Encoding UTF8
    
    Write-Host "Token updated successfully." -ForegroundColor Green
    Show-SystemInfo
}

# System information
function Show-SystemInfo {
    if (-not (Test-Path $CONFIG_FILE)) {
        Write-Host "Not installed, configuration file not found: $CONFIG_FILE" -ForegroundColor Red
        return
    }
    
    $config = Get-Content $CONFIG_FILE -ErrorAction SilentlyContinue
    $settings = @{}
    
    foreach ($line in $config) {
        if ($line -match "^(.+)=(.+)$") {
            $settings[$matches[1]] = $matches[2]
        }
    }
    
    Write-Host "System Information:" -ForegroundColor Cyan
    Write-Host "Service Port: $($settings['PORT'])" -ForegroundColor White
    Write-Host "UUID: $($settings['UUID'])" -ForegroundColor White
    Write-Host "Token: $($settings['TOKEN'])" -ForegroundColor White
    Write-Host "Network Adapter: $($settings['IFACE'])" -ForegroundColor White
    Write-Host "Comment: $($settings['COMMENT'])" -ForegroundColor White
    
    # Show network adapter details
    if ($settings['IFACE']) {
        Write-Host ""
        Write-Host "Selected Network Adapter Details:" -ForegroundColor Yellow
        try {
            $adapter = Get-NetAdapter | Where-Object { $_.Name -eq $settings['IFACE'] }
            if ($adapter) {
                $ip = (Get-NetIPAddress -InterfaceIndex $adapter.InterfaceIndex -AddressFamily IPv4 -ErrorAction SilentlyContinue).IPAddress
                Write-Host "  Adapter Name: $($adapter.Name)" -ForegroundColor White
                Write-Host "  IPv4 Address: $(if($ip) { $ip } else { 'No IPv4' })" -ForegroundColor White
                Write-Host "  Adapter Status: $($adapter.Status)" -ForegroundColor White
                Write-Host "  MAC Address: $($adapter.MacAddress)" -ForegroundColor White
            }
        }
        catch {
            Write-Host "  Failed to get adapter information" -ForegroundColor Red
        }
    }
}

# Run diagnostic
function Start-Diagnostic {
    if (-not (Test-Path $CONFIG_FILE)) {
        Write-Host "Not installed, configuration file not found: $CONFIG_FILE" -ForegroundColor Red
        return
    }
    
    $config = Get-Content $CONFIG_FILE -ErrorAction SilentlyContinue
    $settings = @{}
    
    foreach ($line in $config) {
        if ($line -match "^(.+)=(.+)$") {
            $settings[$matches[1]] = $matches[2]
        }
    }
    
    if (-not $settings['UUID'] -or -not $settings['TOKEN'] -or -not $settings['PORT']) {
        Write-Host "Configuration incomplete, please install service first." -ForegroundColor Red
        return
    }
    
    # Check if service is running
    $port = $settings['PORT']
    try {
        $response = Invoke-WebRequest -Uri "http://localhost:$port/cpu/info?uuid=$($settings['UUID'])`&token=$($settings['TOKEN'])" -TimeoutSec 5 -ErrorAction Stop
        Write-Host "Service is running normally." -ForegroundColor Green
    }
    catch {
        Write-Host "Service is not running, please start service first: python server.py" -ForegroundColor Red
        return
    }
    
    Write-Host "Running diagnostics..." -ForegroundColor Yellow
    $diagnosticUrl = "http://localhost:$port/diagnostic?uuid=$($settings['UUID'])`&token=$($settings['TOKEN'])`&diagnostic=1"
    Write-Host "Diagnostic API URL: $diagnosticUrl" -ForegroundColor Cyan
    
    try {
        $response = Invoke-WebRequest -Uri $diagnosticUrl -TimeoutSec 10
        $diagnosticData = $response.Content | ConvertFrom-Json
        Write-Host "Diagnostic Data:" -ForegroundColor Green
        $diagnosticData | ConvertTo-Json -Depth 10 | Write-Host
    }
    catch {
        Write-Host "Failed to get diagnostic data: $($_.Exception.Message)" -ForegroundColor Red
    }
}

# Start service
function Start-Service {
    if (-not (Test-Path $SERVER_FILE)) {
        Write-Host "Service file not found, please install first." -ForegroundColor Red
        return
    }
    
    Write-Host "Starting WizStudio Status service..." -ForegroundColor Yellow
    try {
        & python $SERVER_FILE
    }
    catch {
        Write-Host "Failed to start service: $($_.Exception.Message)" -ForegroundColor Red
    }
}

# Uninstall service
function Uninstall-Service {
    Write-Host ""
    $confirm = Read-Host "Do you want to delete service and files? (Y/N) [Y]"
    if ($confirm -and $confirm.ToUpper() -ne "Y") {
        Write-Host "Deletion cancelled." -ForegroundColor Yellow
        return
    }
    
    # Delete files
    if (Test-Path $CONFIG_FILE) {
        Remove-Item $CONFIG_FILE -Force
        Write-Host "Configuration file deleted." -ForegroundColor Green
    }
    
    if (Test-Path $SERVER_FILE) {
        Remove-Item $SERVER_FILE -Force
        Write-Host "Service file deleted." -ForegroundColor Green
    }
    
    Write-Host "Service and related files deleted." -ForegroundColor Green
    Write-Host "To reinstall, please run this script again." -ForegroundColor Yellow
}

# Main menu
function Show-MainMenu {
    Show-Title
    Write-Host "1) Install (default)" -ForegroundColor White
    Write-Host "2) Start Service" -ForegroundColor White
    Write-Host "3) View UUID" -ForegroundColor White
    Write-Host "4) Change Token" -ForegroundColor White
    Write-Host "5) System Information" -ForegroundColor White
    Write-Host "6) Run Diagnostic" -ForegroundColor White
    Write-Host "7) Uninstall Service" -ForegroundColor White
    Write-Host ""
    
    $choice = Read-Host "Please select [1-7] (default 1)"
    if ([string]::IsNullOrEmpty($choice)) { $choice = "1" }
    
    switch ($choice) {
        "1" { Install-Service }
        "2" { Start-Service }
        "3" { Show-UUID }
        "4" { Change-Token }
        "5" { Show-SystemInfo }
        "6" { Start-Diagnostic }
        "7" { Uninstall-Service }
        default { Install-Service }
    }
}

# Main program entry
if ($Action -eq "menu") {
    Show-MainMenu
} elseif ($Action -eq "install") {
    Install-Service
} elseif ($Action -eq "start") {
    Start-Service
} else {
    Show-MainMenu
}