# WizStudio Status 启动脚本 - 处理执行策略问题
# 此脚本专门用于解决PowerShell执行策略和数字签名问题

param(
    [switch]$Force
)

# 设置控制台编码
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

Write-Host "=============================== " -ForegroundColor Cyan
Write-Host " WizStudio Status Windows版本 " -ForegroundColor Cyan
Write-Host "=============================== " -ForegroundColor Cyan
Write-Host ""

# 获取脚本目录
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$MainScript = Join-Path $ScriptDir "status_vps.ps1"

# 检查主脚本是否存在
if (-not (Test-Path $MainScript)) {
    Write-Host "错误: 未找到主脚本文件 status_vps.ps1" -ForegroundColor Red
    Write-Host "请确保所有文件都在同一目录中。" -ForegroundColor Yellow
    Read-Host "按回车键退出"
    exit 1
}

# 检查当前执行策略
$currentPolicy = Get-ExecutionPolicy
Write-Host "当前PowerShell执行策略: $currentPolicy" -ForegroundColor Yellow

# 处理不同的执行策略
switch ($currentPolicy) {
    "Restricted" {
        Write-Host "检测到受限执行策略，需要临时绕过..." -ForegroundColor Yellow
        $needBypass = $true
    }
    "AllSigned" {
        Write-Host "检测到AllSigned策略，脚本未签名，需要临时绕过..." -ForegroundColor Yellow
        $needBypass = $true
    }
    "RemoteSigned" {
        Write-Host "检测到RemoteSigned策略，本地脚本应该可以运行..." -ForegroundColor Green
        $needBypass = $false
    }
    "Unrestricted" {
        Write-Host "检测到不受限策略，脚本可以正常运行..." -ForegroundColor Green
        $needBypass = $false
    }
    "Bypass" {
        Write-Host "检测到Bypass策略，脚本可以正常运行..." -ForegroundColor Green
        $needBypass = $false
    }
    default {
        Write-Host "未知执行策略: $currentPolicy" -ForegroundColor Yellow
        $needBypass = $true
    }
}

Write-Host ""

if ($needBypass -and -not $Force) {
    Write-Host "解决方案选项:" -ForegroundColor Cyan
    Write-Host "1. 临时绕过执行策略 (推荐，安全)" -ForegroundColor White
    Write-Host "2. 永久更改为RemoteSigned策略 (需要管理员权限)" -ForegroundColor White
    Write-Host "3. 取消运行" -ForegroundColor White
    Write-Host ""
    
    do {
        $choice = Read-Host "请选择 [1-3] (默认1)"
        if ([string]::IsNullOrEmpty($choice)) { $choice = "1" }
    } while ($choice -notmatch '^[1-3]$')
    
    switch ($choice) {
        "1" {
            Write-Host "使用临时绕过策略运行脚本..." -ForegroundColor Green
            Write-Host "注意: 这不会永久更改系统设置，仅对当前会话有效。" -ForegroundColor Yellow
            Write-Host ""
            
            try {
                & powershell.exe -ExecutionPolicy Bypass -File $MainScript
            }
            catch {
                Write-Host "运行脚本时出错: $($_.Exception.Message)" -ForegroundColor Red
            }
        }
        "2" {
            Write-Host "尝试永久更改执行策略..." -ForegroundColor Yellow
            try {
                Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope LocalMachine -Force
                Write-Host "执行策略已更改为RemoteSigned" -ForegroundColor Green
                Write-Host "现在运行主脚本..." -ForegroundColor Green
                Write-Host ""
                & $MainScript
            }
            catch {
                Write-Host "更改执行策略失败: $($_.Exception.Message)" -ForegroundColor Red
                Write-Host "可能需要以管理员身份运行PowerShell。" -ForegroundColor Yellow
                Write-Host "尝试使用临时绕过策略..." -ForegroundColor Yellow
                Write-Host ""
                
                try {
                    & powershell.exe -ExecutionPolicy Bypass -File $MainScript
                }
                catch {
                    Write-Host "运行脚本时出错: $($_.Exception.Message)" -ForegroundColor Red
                }
            }
        }
        "3" {
            Write-Host "已取消运行。" -ForegroundColor Yellow
            Read-Host "按回车键退出"
            exit 0
        }
    }
} else {
    Write-Host "直接运行主脚本..." -ForegroundColor Green
    Write-Host ""
    
    try {
        if ($needBypass -or $Force) {
            & powershell.exe -ExecutionPolicy Bypass -File $MainScript
        } else {
            & $MainScript
        }
    }
    catch {
        Write-Host "运行脚本时出错: $($_.Exception.Message)" -ForegroundColor Red
        Write-Host "尝试使用绕过策略..." -ForegroundColor Yellow
        
        try {
            & powershell.exe -ExecutionPolicy Bypass -File $MainScript
        }
        catch {
            Write-Host "仍然无法运行: $($_.Exception.Message)" -ForegroundColor Red
        }
    }
}

Write-Host ""
Read-Host "按回车键退出"