#!/usr/bin/env python3
# -*- coding: utf-8 -*-
import json
import os
import platform
import subprocess
import time
import psutil
from http.server import BaseHTTPRequestHandler, HTTPServer
from urllib.parse import urlparse, parse_qs

CONFIG_FILE = "wizstudio_status.conf"

def read_config():
    cfg = {}
    try:
        if os.path.exists(CONFIG_FILE):
            with open(CONFIG_FILE, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    if "=" in line:
                        k, v = line.split("=", 1)
                        cfg[k.strip()] = v.strip()
    except Exception as e:
        print(f"Error reading config: {e}")
    return cfg

def human_gb(b, digits=2):
    try:
        return f"{round(b / (1024**3), digits)}GB"
    except Exception:
        return "0GB"

def human_percent(v, digits=1):
    try:
        return f"{round(v, digits)}%"
    except Exception:
        return "0%"

def human_bitrate(bps):
    try:
        if bps >= 1_000_000:
            return f"{round(bps/1_000_000.0, 2)}Mbps"
        else:
            return f"{round(bps/1000.0, 2)}Kbps"
    except Exception:
        return "0.00Kbps"

def cpu_info():
    try:
        # Get CPU information
        cpu_name = platform.processor()
        if not cpu_name or cpu_name.strip() == "":
            # Try to get from registry
            try:
                import winreg
                key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, 
                    r"HARDWARE\DESCRIPTION\System\CentralProcessor\0")
                cpu_name = winreg.QueryValueEx(key, "ProcessorNameString")[0]
                winreg.CloseKey(key)
            except:
                cpu_name = "Unknown CPU"
        
        # Get core count
        cores = psutil.cpu_count(logical=False)
        threads = psutil.cpu_count(logical=True)
        
        # Get frequency
        try:
            freq = psutil.cpu_freq()
            if freq and freq.max:
                freq_str = f"@{round(freq.max / 1000.0, 2)}GHz"
            else:
                freq_str = ""
        except:
            freq_str = ""
        
        core_part = f" {cores}Cores" if cores else ""
        if threads and threads != cores:
            core_part += f"/{threads}Threads"
        
        return f"{cpu_name.strip()}{core_part}{freq_str}"
    except Exception as e:
        return f"Unknown CPU (Error: {e})"

def cpu_usage_percent():
    try:
        return psutil.cpu_percent(interval=1.0)
    except Exception:
        return 0.0

def mem_usage():
    try:
        mem = psutil.virtual_memory()
        return (mem.used, mem.total)
    except Exception:
        return (0, 0)

def disk_usage():
    try:
        total_used = 0
        total_size = 0
        
        # Get all disk partitions
        partitions = psutil.disk_partitions()
        for partition in partitions:
            try:
                if partition.fstype:  # Only count partitions with filesystem
                    usage = psutil.disk_usage(partition.mountpoint)
                    total_used += usage.used
                    total_size += usage.total
            except (PermissionError, FileNotFoundError):
                continue
        
        return (total_used, total_size)
    except Exception:
        return (0, 0)

def disk_io():
    try:
        # First sample
        io1 = psutil.disk_io_counters()
        if not io1:
            return (0.0, 0.0, 0, 0)
        
        time.sleep(1.0)
        
        # Second sample
        io2 = psutil.disk_io_counters()
        if not io2:
            return (0.0, 0.0, 0, 0)
        
        # Calculate differences
        read_bytes = max(0, io2.read_bytes - io1.read_bytes)
        write_bytes = max(0, io2.write_bytes - io1.write_bytes)
        read_count = max(0, io2.read_count - io1.read_count)
        write_count = max(0, io2.write_count - io1.write_count)
        
        # Convert to MB/s
        read_mbps = read_bytes / (1024 * 1024)
        write_mbps = write_bytes / (1024 * 1024)
        
        return (read_mbps, write_mbps, read_count, write_count)
    except Exception:
        return (0.0, 0.0, 0, 0)

def network_io(interface_name):
    try:
        # Get network interface statistics
        net_io = psutil.net_io_counters(pernic=True)
        
        # Find matching interface
        target_interface = None
        for iface_name, stats in net_io.items():
            if interface_name.lower() in iface_name.lower():
                target_interface = stats
                break
        
        if not target_interface:
            # If specified interface not found, use total
            target_interface = psutil.net_io_counters()
        
        if not target_interface:
            return (0, 0)
        
        # First sample
        bytes_sent1 = target_interface.bytes_sent
        bytes_recv1 = target_interface.bytes_recv
        
        time.sleep(2.0)
        
        # Second sample
        if interface_name:
            net_io2 = psutil.net_io_counters(pernic=True)
            target_interface2 = None
            for iface_name, stats in net_io2.items():
                if interface_name.lower() in iface_name.lower():
                    target_interface2 = stats
                    break
            if not target_interface2:
                target_interface2 = psutil.net_io_counters()
        else:
            target_interface2 = psutil.net_io_counters()
        
        if not target_interface2:
            return (0, 0)
        
        bytes_sent2 = target_interface2.bytes_sent
        bytes_recv2 = target_interface2.bytes_recv
        
        # Calculate rate (bps)
        upload_bps = max(0, (bytes_sent2 - bytes_sent1) * 8.0 / 2.0)
        download_bps = max(0, (bytes_recv2 - bytes_recv1) * 8.0 / 2.0)
        
        return (upload_bps, download_bps)
    except Exception as e:
        print(f"Network IO error: {e}")
        return (0, 0)

def get_uptime_formatted():
    try:
        # Windows system boot time
        boot_time = psutil.boot_time()
        uptime_seconds = time.time() - boot_time
        
        # Convert to time units
        total_minutes = int(uptime_seconds // 60)
        total_hours = total_minutes // 60
        total_days = total_hours // 24
        years = total_days // 365
        months = (total_days % 365) // 30
        
        minutes = total_minutes % 60
        hours = total_hours % 24
        days = total_days % 30
        
        # Build formatted string
        parts = []
        if years > 0:
            parts.append(f"{years}y")
        if months > 0:
            parts.append(f"{months}m")
        if days > 0:
            parts.append(f"{days}d")
        if hours > 0:
            parts.append(f"{hours}h")
        if minutes > 0:
            parts.append(f"{minutes}min")
        
        if parts:
            return "".join(parts)
        else:
            return "less than 1min"
    except Exception as e:
        return f"N/A (Error: {e})"

class Handler(BaseHTTPRequestHandler):
    server_version = "WizStudioStatus/1.0"

    def _send(self, code, payload_str):
        if isinstance(payload_str, dict):
            body = json.dumps(payload_str, ensure_ascii=False).encode("utf-8")
        else:
            body = json.dumps(payload_str, ensure_ascii=False).encode("utf-8")
        
        self.send_response(code)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(body)))
        self.end_headers()
        self.wfile.write(body)

    def do_GET(self):
        cfg = read_config()
        req = urlparse(self.path)
        q = parse_qs(req.query)
        uuid = q.get("uuid", [""])[0]
        token = q.get("token", [""])[0]

        want_uuid = cfg.get("UUID", "")
        want_token = cfg.get("TOKEN", "")
        iface = cfg.get("IFACE", "")

        # Check diagnostic mode
        is_diagnostic = "diagnostic" in q and q["diagnostic"][0].lower() in ("1", "true", "yes")
        
        # Authentication
        if not uuid or not token:
            self._send(400, "Missing uuid or token")
            return
        if uuid != want_uuid or token != want_token:
            self._send(401, "Unauthorized")
            return

        path = req.path.rstrip("/")
        try:
            # Diagnostic endpoint
            if path == "/diagnostic" and is_diagnostic:
                diagnostic_data = self._collect_diagnostic_data()
                self._send(200, diagnostic_data)
                return
                
            # Standard API endpoints
            if path == "/cpu/info":
                val = cpu_info()
                self._send(200, val)
            elif path == "/cpu/usage":
                val = f"{round(cpu_usage_percent(), 1)}%"
                self._send(200, val)
            elif path == "/ram/usage0":
                used, total = mem_usage()
                val = f"{human_gb(used, 2)}/{human_gb(total, 2)}"
                self._send(200, val)
            elif path == "/ram/usage1":
                used, total = mem_usage()
                pct = (used/total*100.0) if total > 0 else 0.0
                val = human_percent(pct, 1)
                self._send(200, val)
            elif path == "/disk/usage0":
                used, total = disk_usage()
                val = f"{human_gb(used, 1)}/{human_gb(total, 1)}"
                self._send(200, val)
            elif path == "/disk/usage1":
                used, total = disk_usage()
                pct = (used/total*100.0) if total > 0 else 0.0
                val = f"{round(pct, 1)}%"
                self._send(200, val)
            elif path == "/disk/io":
                read_mbps, write_mbps, read_iops, write_iops = disk_io()
                val = {
                    "read": f"{round(read_mbps, 2)}MB/s",
                    "write": f"{round(write_mbps, 2)}MB/s",
                    "read_iops": read_iops,
                    "write_iops": write_iops
                }
                self._send(200, val)
            elif path == "/network/status":
                if not iface:
                    self._send(500, "No IFACE configured")
                    return
                
                upload_bps, download_bps = network_io(iface)
                val = f"{human_bitrate(upload_bps)}/{human_bitrate(download_bps)}"
                self._send(200, val)
            elif path == "/time":
                val = get_uptime_formatted()
                self._send(200, val)
            else:
                self._send(404, "Not Found")
        except Exception as e:
            self._send(500, f"Error: {e}")

    def _collect_diagnostic_data(self):
        """Collect diagnostic data for troubleshooting"""
        data = {
            "version": "1.0",
            "timestamp": time.time(),
            "system_info": {},
            "cpu": {},
            "memory": {},
            "disk": {},
            "network": {},
            "available_tools": {},
            "debug_info": {}
        }
        
        # System information
        try:
            data["system_info"]["hostname"] = platform.node()
            data["system_info"]["platform"] = platform.platform()
            data["system_info"]["system"] = platform.system()
            data["system_info"]["release"] = platform.release()
            data["system_info"]["version"] = platform.version()
            data["system_info"]["machine"] = platform.machine()
            data["system_info"]["processor"] = platform.processor()
        except Exception as e:
            data["system_info"]["error"] = str(e)
            
        # CPU information
        try:
            data["cpu"]["info"] = cpu_info()
            data["cpu"]["usage"] = f"{round(cpu_usage_percent(), 1)}%"
            data["cpu"]["count_logical"] = psutil.cpu_count(logical=True)
            data["cpu"]["count_physical"] = psutil.cpu_count(logical=False)
        except Exception as e:
            data["cpu"]["error"] = str(e)
            
        # Memory information
        try:
            used, total = mem_usage()
            data["memory"]["used"] = human_gb(used, 2)
            data["memory"]["total"] = human_gb(total, 2)
            data["memory"]["percent"] = human_percent((used/total*100.0) if total > 0 else 0.0, 1)
        except Exception as e:
            data["memory"]["error"] = str(e)
            
        # Disk information
        try:
            used, total = disk_usage()
            data["disk"]["used"] = human_gb(used, 1)
            data["disk"]["total"] = human_gb(total, 1)
            data["disk"]["percent"] = f"{round((used/total*100.0) if total > 0 else 0.0, 1)}%"
            
            # Disk partition information
            partitions = []
            for partition in psutil.disk_partitions():
                try:
                    usage = psutil.disk_usage(partition.mountpoint)
                    partitions.append({
                        "device": partition.device,
                        "mountpoint": partition.mountpoint,
                        "fstype": partition.fstype,
                        "total": human_gb(usage.total, 1),
                        "used": human_gb(usage.used, 1),
                        "free": human_gb(usage.free, 1)
                    })
                except (PermissionError, FileNotFoundError):
                    continue
            data["disk"]["partitions"] = partitions
        except Exception as e:
            data["disk"]["error"] = str(e)
            
        # Network information
        try:
            cfg = read_config()
            iface = cfg.get("IFACE", "")
            data["network"]["configured_iface"] = iface
            
            # Network interface information
            interfaces = []
            net_io = psutil.net_io_counters(pernic=True)
            for name, stats in net_io.items():
                interfaces.append({
                    "name": name,
                    "bytes_sent": stats.bytes_sent,
                    "bytes_recv": stats.bytes_recv,
                    "packets_sent": stats.packets_sent,
                    "packets_recv": stats.packets_recv
                })
            data["network"]["interfaces"] = interfaces
        except Exception as e:
            data["network"]["error"] = str(e)
            
        # Available tools check
        data["available_tools"]["psutil"] = True
        data["available_tools"]["platform"] = True
        
        # Debug information
        try:
            data["debug_info"]["python_version"] = platform.python_version()
            data["debug_info"]["psutil_version"] = psutil.__version__
        except Exception:
            pass
            
        return data

    def log_message(self, fmt, *args):
        return

def run():
    cfg = read_config()
    port = int(cfg.get("PORT", "8008") or "8008")
    addr = ("0.0.0.0", port)
    httpd = HTTPServer(addr, Handler)
    print(f"WizStudio Status Server started on port {port}")
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        print("\nServer stopped")
    finally:
        httpd.server_close()

if __name__ == "__main__":
    run()